<?php

namespace App\Http\Controllers;

use App\Models\account;
use App\Models\agentTarget;
use App\Models\client;
use App\Models\department;
use App\Models\leadMonitor;
use App\Models\personalInformation;
use App\Models\project;
use App\Models\projectCost;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Models\Role;

use App\Models\User;
use Carbon\Carbon;

class OperationController extends Controller
{
    public function authenticate(Request $request)
    {    
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);
        
        if (Auth::attempt($credentials)) {
        
            $user = Auth::user();
            
            if($user->status == 1)
            {
                $request->session()->regenerate();
                if ($user->hasRole('admin'))
                {
                    return redirect()->route('admin.departments');
                }
                else if ($user->hasRole('manager'))
                {
                    return back()->with(['message' => 'Manager Login Success.']);
                }
                else
                {
                    return back()->with(['error' => 'Invalid email or password.']);
                }
            }
            else 
            {
                return back()->with(['error' => 'Your account is deactive.']);
            }
        }
        else
        {
            return back()->with(['error' => 'Invalid email or password.']);            
        }
    }
    
    public function logout(Request $request)
    {
        Auth::logout();
        
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        return redirect()->route('login');
    }

    public function createAccount(Request $request)
    {
        $validated = $request->validate([

            'firstname' => 'required|regex:/^[ a-zA-Z ]+$/u',
            'lastname' => 'required|regex:/^[ a-zA-Z ]+$/u',
            'age' => 'required',
            'cellphone' => 'required',
            'cellphone-alt' => 'required',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6',

        ]);

        if($request->hasFile('avatar'))
        {
            $filename = time().'.'.request()->avatar->getClientOriginalExtension();
            $request->avatar->move(public_path('uploads/profile/'), $filename);
            $avatar = '/uploads/profile/'.$filename;

            $user = User::create([

                'username' => substr($request->input('email'), 0, strrpos($request->input('email'), '@')),
                'email' => $request->input('email'),
                'password' => Hash::make($request->input('password')),
    
            ]);
    
            $role = Role::where('name' , 'manager')->first();
            $user->assignRole([$role->id]);
            
            $personal_information = personalInformation::create([
    
                'profile_image' => $avatar,
                'name' => $request->input('first-name') .' '. $request->input('last-name'),
                'cell_phone' => $request->input('cellphone'),
                'cell_phone_alt' => $request->input('cellphone-alt'),
                'department_id' => $request->input('department-id'),
                'age' => $request->input('age'),
                'gender' => $request->input('gender'),
                'user_id' => $user->id
    
            ]);
            
            return back()->with(['message' => 'User regirsted successfully.']);
        }
        else
        {   
            return back()->with(['error' => 'Profile Picture is required.']);
        }
    }

    public function storeDepartment(Request $request)
    {
        $validated = $request->validate([

            'department-name' => 'required|regex:/^[ a-zA-Z ]+$/u',

        ]);

        $department = department::create([
            
            'name' => $request->input('department-name'),

        ]);

        if($department)
        {
            return back()->with(['message' => 'Department added successfully.']);
        }
    }

    public function storeAccount(Request $request)
    {
        $validated = $request->validate([

            'account-title' => 'required|regex:/^[ a-zA-Z ]+$/u',

        ]);

        $account = account::create([
            
            'title' => $request->input('account-title'),

        ]);

        if($account)
        {
            return back()->with(['message' => 'Account added successfully.']);
        }
    }
    
    public function storeLead(Request $request)
    {
        $validated = $request->validate([

            'client-name' => 'required|regex:/^[ a-zA-Z ]+$/u',
            'business-name' => 'required',
            'client-email' => 'required|email|unique:users,email',
            'client-cellphone' => 'required',
            'client-cellphone-alt' => 'required',
            'client-location' => 'required',
            'onboard-date' => 'required',
            'account_id' => 'required',
            'agent-id' => 'required'

        ]);
        
        $client = client::create([
            
            'name' => $request->input('client-name'),
            'business_name' => $request->input('business-name'),
            'email' => $request->input('client-email'),
            'cell_phone' => $request->input('client-cellphone'),
            'cell_phone_alt' => $request->input('client-cellphone-alt'),
            'location' => $request->input('client-location'),
            'on_board_date' => $request->input('onboard-date'),
            'account_id' => $request->input('account_id'),
            'created_by' => $request->input('agent-id'),

        ]);

        if($client)
        {
            $leadStatus = leadMonitor::create([
            
                'lead_id' => $client->id,
                'status_id' => 1,
                // 'created_by' => Auth::user()->id,
                'created_by' => $request->input('agent-id'),
                'date' => Carbon::now()->format('Y-m-d')
    
            ]);
            
            if($leadStatus)
            {
                return back()->with(['message' => 'Lead added successfully.']);
            }

        }
    }

    public function departmentStatus($id)
    {
        $department = department::findOrFail($id);
        if($department->status == 0)
        {
            $department->status = 1;
            $department->save();
            return back()->with(['message' => 'Department Active successfully.']);
        }
        else
        {
            $department->status = 0;
            $department->save();
            return back()->with(['error' => 'Department Deactive successfully.']);
        }
    }

    public function accountStatus($id)
    {
        $account = account::findOrFail($id);
        if($account->status == 0)
        {
            $account->status = 1;
            $account->save();
            return back()->with(['message' => 'Account Active successfully.']);
        }
        else
        {
            $account->status = 0;
            $account->save();
            return back()->with(['error' => 'Account Deactive successfully.']);
        }
    }

    public function leadConverted(Request $request)
    {
        $validated = $request->validate([

            'project-title' => 'required',
            'kt_docs_tinymce_basic' => 'required',
            'project-budget' => 'required',
            'amount-charge' => 'required',
            'currency' => 'required',
            'payment-through' => 'required'

        ]);
        
        $project = project::create([
            
            'title'                 => $request->input('project-title'),
            'description'           => $request->input('kt_docs_tinymce_basic'),
            'budget'                => $request->input('project-budget'),
            'currency'              => $request->input('currency'),
            'client_id'             => $request->input('client_id'),
            'comment'               => $request->input('comments') ?? 'No Comments'

        ]);
        
        if($project)
        {
            $projectCost = projectCost::create([
            
                'project_id'        => $project->id,
                'amount_charged'    => $request->input('amount-charge'),
                'amount_type'       => 'Front',
                'pay_through'       => $request->input('payment-through'),
                'comments'          => $request->input('comments') ?? 'No Comments',
                'date'              => Carbon::now()->format('Y-m-d'),
                'user_id'           => $request->input('agent_id')
    
            ]);

            if($projectCost)
            {
                $convert = leadMonitor::create([

                    'lead_id' => $request->input('client_id'),
                    'status_id' => 7,
                    'created_by' => $request->input('agent_id'),
                    'date' => Carbon::now()->format('Y-m-d'),
                    'comments' => $request->input('comments') ?? 'No Comments'

                ]);

                if($convert)
                {
                    $lead = client::find($request->input('client_id'));
                    $lead->lead_status = 7;
                    if($lead->save()){
                        return back()->with(['message' => 'Lead converted successfully.']);
                    }
                }
            }
        }
        else
        {
            return back()->with(['error' => 'Lead did not converted.']);
        }
    }

    public function leadStatus(Request $request)
    {
        $convert = leadMonitor::create([

            'lead_id' => $request->input('client_id'),
            'status_id' => $request->input('lead_status'),
            'created_by' => $request->input('agent_id'),
            'date' => Carbon::now()->format('Y-m-d'),
            'comments' => $request->input('comments') ?? 'No Comments'

        ]);

        if($convert)
        {
            $lead = client::find($request->input('client_id'));
            $lead->lead_status = $request->input('lead_status');
            
            if($lead->save()){

                return back()->with(['message' => 'Lead status changed successfully.']);
                
            }

        }
    }

    public function storeProject(Request $request)
    {
        $validated = $request->validate([

            'project-title' => 'required',
            'kt_docs_tinymce_basic' => 'required',
            'project-budget' => 'required',
            'amount-charge' => 'required',
            'currency' => 'required',
            'payment-through' => 'required',
            'client_id' => 'required',
            'agent_id' => 'required',

        ]);
        
        $project = project::create([
            
            'title'                 => $request->input('project-title'),
            'description'           => $request->input('kt_docs_tinymce_basic'),
            'budget'                => $request->input('project-budget'),
            'currency'              => $request->input('currency'),
            'client_id'             => $request->input('client_id'),
            'comment'               => $request->input('comments') ?? 'N/A',

        ]);
        
        if($project)
        {
            $projectCost = projectCost::create([
            
                'project_id'        => $project->id,
                'amount_charged'    => $request->input('amount-charge'),
                'amount_type'       => 'front',
                'pay_through'       => $request->input('payment-through'),
                'comments'          => $request->input('comments') ?? 'N/A',
                'date'              => now(),
                'user_id'           => $request->input('agent_id'),
    
            ]);
            return back()->with(['message' => 'Project added successfully.']);
        }
        else
        {
            return back()->with(['error' => 'Project did not add.']);
        }
    }

    public function storeProjectCosting(Request $request)
    {   
        $validated = $request->validate([

            'project_id' => 'required',
            'amount-charged' => 'required',
            'cost_type' => 'required',
            'kt_docs_tinymce_basic' => 'required',
            'user_id' => 'required',
            'cost-date' => 'required'

        ]);
        
        $projectCost = projectCost::create([
            
            'project_id' => $request->input('project_id'),
            'amount_charged' => $request->input('amount-charged'),
            'amount_type' => $request->input('cost_type'),
            'pay_through' => $request->input('payment-through'),
            'comments' => $request->input('kt_docs_tinymce_basic'),
            'date' => $request->input('cost-date'),
            'user_id' => $request->input('user_id'),

        ]);
        
        if($projectCost)
        {
            return back()->with(['message' => 'New charge created successfully.']);
        }
        else
        {
            return back()->with(['error' => 'Cost did not add.']);
        }
    }

    public function storeAgent(Request $request)
    {
        
    }

    public function storeTarget(Request $request)
    {
        $validated = $request->validate([

            'agent-target' => 'required',
            'month-id' => 'required'

        ]);

        $target = agentTarget::create([
            
            'amount' => $request->input('agent-target'),
            'month' => $request->input('month-id'),
            'user_id' => $request->input('agent-id')

        ]);

        if($target)
        {
            return back()->with(['message' => 'Target set successfully.']);
        }
        else
        {
            return back()->with(['error' => 'Something went wrong.']);
        }
    }

}